// Copyright 1984-2019 The MathWorks, Inc.
// All Rights Reserved.

import java.net.URL;
import java.io.IOException;
import com.mathworks.mps.client.MWClient;
import com.mathworks.mps.client.MWHttpClient;
import com.mathworks.mps.client.MATLABException;

/**
 * The interface to the remote magic square service.
 */
interface MatlabMagic {
    /**
      * Calculate magic square of a given size.
      * @param size  how big is the square on each size
      * @return a two-dimensional array, [size][size] -- the magic square
      * @throw MATLABException if a MATLAB error occurred
      * @throw IOException if a communication error prevented the request from completing
      */    
    double[][] mymagic(int size) throws IOException, MATLABException;
}

public class Magic {
    
    /**
     * @params args Should be length 2; first is URL of the magic square component on the server,
     *              second is the size of the magic square to calculate     
     */
    public static void main(String[] args){
    
        // Create a non-interruptible MWHttpClient instance
        final MWClient client = new MWHttpClient();
        
        try{
            // Create the proxy object that represents magic.ctf
            final MatlabMagic m = client.createProxy( new URL(args[0]), MatlabMagic.class );            
            
            // The proxy object has mymagic as one of its public methods. Invocation of mymagic
            // results in a server request that gets the magic square in response
            final double[][] result = m.mymagic( Integer.parseInt( args[1] ) );
            
            // Let's print the magic square
            printResult(result);
            
        }catch(MATLABException ex){
        
            // This exception represents errors in MATLAB and provides useful information
            // like the MATLAB stack trace or the error ID associated with this error.
            System.out.println(ex);            
        }catch(IOException ex){
        
            // This exception can represent network issues. It is also thrown when the 
            // HTTP response received from the server has a status of 4XX or 5XX
            System.out.println(ex);                    
        }finally{
        
            // We should close the client when we know that we are not going to need it any more
            // Once the client is closed, an exception will be thrown if a MATLAB function is 
            // invoked using the proxy object that was created using client.
            client.close();        
        }
    }
    
    /**
      * Print the magic square
      * @param result magic square to be printed      
      */
    private static void printResult(double[][] result){
        if( result != null ){
            for(double[] row : result){
                for(double element : row){
                    System.out.print(element + " ");
                }
                System.out.println();
            }
        }else{
            System.out.println("Empty magic square returned from MATLAB. This is possible if 0 was passed as input to MATLAB.");        
        }
    }
}
